<?php

namespace VisualComposer\Modules\Vendors\Plugins;

if (!defined('ABSPATH')) {
    header('Status: 403 Forbidden');
    header('HTTP/1.1 403 Forbidden');
    exit;
}

use VisualComposer\Framework\Container;
use VisualComposer\Framework\Illuminate\Support\Module;
use VisualComposer\Helpers\Frontend;
use VisualComposer\Helpers\Traits\EventsFilters;
use VisualComposer\Helpers\Traits\WpFiltersActions;
use VisualComposer\Modules\Editors\Settings\TitleController;

/**
 * Backward compatibility with "Yoast" plugin.
 *
 * @see https://wordpress.org/plugins/wordpress-seo/
 */
class YoastController extends Container implements Module
{
    use EventsFilters;
    use WpFiltersActions;

    /**
     * @var TitleController
     */
    protected $titleController;

    public function __construct()
    {
        $this->wpAddAction('plugins_loaded', 'initialize');
    }

    protected function initialize()
    {
        if (!function_exists('YoastSEO')) {
            return;
        }
        /** @see \VisualComposer\Modules\Vendors\Plugins\YoastController::initializeYoast */
        $this->addEvent('vcv:inited', 'initializeYoast', 11);

        /** @see \VisualComposer\Modules\Vendors\Plugins\YoastController::overrideFrontendOutput */
        $this->wpAddFilter(
            'wpseo_frontend_presentation',
            'overrideFrontendOutput',
            10,
            1
        );

        $this->titleController = vcapp('EditorsSettingsTitleController');
        $this->preventRemovingTitleFromBreadcrumbs();

        /** @see \VisualComposer\Modules\Vendors\Plugins\YoastController::fixYoastPremiumBreadcrumbs */
        if (defined('WPSEO_PREMIUM_FILE')) {
            $this->wpAddFilter(
                'wpseo_frontend_presentation',
                'fixYoastPremiumBreadcrumbs',
                10,
                1
            );
        }
    }

    protected function initializeYoast(Frontend $frontendHelper)
    {
        if (isset($GLOBALS['wpseo_metabox']) && $frontendHelper->isFrontend()) {
            $this->removeFeScript();
        }

        if (!isset($GLOBALS['post_ID']) && $frontendHelper->isFrontend()) {
            $this->setGlobalPostId();
        }
    }

    protected function removeFeScript()
    {
        if (isset($GLOBALS['wpseo_metabox'])) {
            remove_action('admin_enqueue_scripts', [$GLOBALS['wpseo_metabox'], 'enqueue']);
        }
    }

    protected function setGlobalPostId()
    {
        $requestHelper = vchelper('Request');
        $globalsHelper = vchelper('Globals');
        $globalsHelper->set('post_ID', $requestHelper->input('vcv-source-id'));
    }

    /**
     * For our layout temples where we completely rewrite global query we need return it for initial pages.
     *
     * @param array $contextPresentation
     *
     * @return array
     */
    protected function overrideFrontendOutput($contextPresentation)
    {
        // @codingStandardsIgnoreStart
        global $wp_query, $wp_the_query;

        if (
            empty($wp_query->query['queriedPage'])
            || empty($wp_the_query->query['queriedPage'])
            || empty($contextPresentation->model)
        ) {
            return $contextPresentation;
        }

        $globalsHelper = vchelper('Globals');
        $globalsHelper->backup('yoast.wp_query', 'wp_query');
        $globalsHelper->backup('yoast.wp_the_query', 'wp_the_query');
        $copyWpQuery = $globalsHelper->get('wp_query');
        $copyWpTheQuest = $globalsHelper->get('wp_the_query');

        // set globals
        $globalsHelper->set('wp_query', $copyWpQuery->query['queriedPage']);
        $globalsHelper->set('wp_the_query', $copyWpTheQuest->query['queriedPage']);

        $model = YoastSEO()->meta->for_current_page()->indexable;

        if ($model) {
            $contextPresentation->model = $model;
        }

        // restore globals
        $globalsHelper->restore('yoast.wp_query', 'wp_query');
        $globalsHelper->restore('yoast.wp_the_query', 'wp_the_query');

        return $contextPresentation;
    }

    /**
     * Prevent removing the title from breadcrumbs generated by YoastSEO plugin
     *
     * Why PHP_INT_MAX constant used in calculations for priority argument?
     * @see \Yoast\WP\SEO\Integrations\Front_End\WP_Robots_Integration::register_hooks()
     */
    protected function preventRemovingTitleFromBreadcrumbs()
    {
        $this->wpAddFilter(
            'wp_robots',
            $this->titleController->removeTitleFilterClosure
        );
        $this->wpAddFilter(
            'wp_robots',
            $this->titleController->addTitleFilterClosure,
            PHP_INT_MAX - 9
        );
    }

    /**
     * Fix yoast breadcrumbs issue with our HFS post types.
     *
     * @param object $presentation
     *
     * @return object
     */
    protected function fixYoastPremiumBreadcrumbs($presentation)
    {
        $hfsList = [
            'vcv_headers',
            'vcv_footers',
            'vcv_sidebars',
        ];

        $postType = get_post_type();
        $isCurrentYoastPostTypeisOurHfs = $postType !== $presentation->model->object_sub_type && !in_array($postType, $hfsList);

        if ($isCurrentYoastPostTypeisOurHfs) {
            $presentation->model->object_sub_type = $postType;
            $presentation->model->permalink = get_permalink();
            $presentation->model->object_id = get_the_ID();
            $presentation->model->breadcrumb_title = get_the_title();
        }

        return $presentation;
    }
}
